#include "Config.h"
/******************************************************************************
 * Archive.cpp - Archive                                                      *
 ******************************************************************************
 * Projet       : Archive(R)                                                  *
 * Auteur       : Arnaud Storq (http://norecess.planet-d.net)                 *
 ******************************************************************************/
#include "Archive.h"

// Lisez le contenu du fichier "Archive.h" pour plus d'informations

// ---------------------------------------------------------------------------- CONSTANTES
#define ARCHIVE_SIGNATURE       "CDR1"  // doit etre long de 4 octets

// ---------------------------------------------------------------------------- CONSTRUCTEUR
Archive::Archive()
{
}

// ---------------------------------------------------------------------------- DESTRUCTEUR
Archive::~Archive()
{
    // Libere la mmoire restante si ce n'est pas deja fait!
    archiveVider();
}

// ---------------------------------------------------------------------------- VIDER CONTENU ARCHIVE
ArchiveResultat Archive::archiveVider()
{
    m_iterateur = m_elements.begin();
    
    while (m_iterateur!=m_elements.end())
    {
        ArchiveElement element = *m_iterateur;
            
        delete [] element.pointeur;
            
        m_iterateur = m_elements.erase(m_iterateur); // "erase" efface l'element courrant et retourne le prochain element !
    }
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- OUVRIR ARCHIVE
ArchiveResultat Archive::archiveOuvrir(string &nomFichier)
{
    archiveVider();
    
    // Ouvre le fichier d'archive en lecture au format binaire ("rb" pour Read Binary").
    // L'ouvrir en format lecture ("r") poserait des problemes avec les caracteres de fin
    // de ligne, par exemple.
    FILE *fichier = fopen(nomFichier.c_str(), "rb");
    
    // L'archive specifie n'existe pas!!
    if (fichier==NULL)
    {
        return ID_ARCHIVE_ERREUR_FICHIER_INEXISTANT;
    }
    
    // Lecture de l'entete
    ArchiveEntete entete;
    fread(&entete, sizeof(ArchiveEntete), 1, fichier);
    
    // Verification de la signature
    // Cela permet de s'asssurer que le fichier que l'on s'appete  relir
    //  bien t gener avec l'outil.
    if (entete.signature[0]!=ARCHIVE_SIGNATURE[0] ||
        entete.signature[1]!=ARCHIVE_SIGNATURE[1] ||
        entete.signature[2]!=ARCHIVE_SIGNATURE[2] ||
        entete.signature[3]!=ARCHIVE_SIGNATURE[3])
    {
        fclose(fichier);
        
        return ID_ARCHIVE_ERREUR_MAUVAISE_VERSION;
    }
    
    // Lecture du contenu de l'archive
    for (int i=0; i<entete.nombreFichiers; i++)    
    {
        // Lecture d'un element 
        ArchiveElement element;
        fread(&element, sizeof(ArchiveElement), 1, fichier);
            
        // Lecture du contenu de l'element (donnes compresses du fichier)
        switch (element.typeCompression)
        {
            case ID_ARCHIVE_COMPRESSION_AUCUNE:
            {
                element.tailleCompressee = 0;
                element.pointeur = new char[element.tailleOriginale];
            
                fread(element.pointeur, element.tailleOriginale, 1, fichier);
                
                break;
            };
            
            case ID_ARCHIVE_COMPRESSION_ZLIB:
            {
                element.pointeur = new char[element.tailleCompressee];
            
                fread(element.pointeur, element.tailleCompressee, 1, fichier);
                
                break;
            };
        }
        
        // Rajouter l'element a notre vecteur STL d'elements !
        m_elements.push_back(element);
    }
    
    fclose(fichier);
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- SAUVEGARDER SOUS ARCHIVE
ArchiveResultat Archive::archiveSauvegarder(string &nomFichier)
{
    // Ouvre le fichier en ecriture binaire ("wb").    
    FILE *fichier = fopen(nomFichier.c_str(), "wb");
    
    // Ecriture de l'entete
    {
        ArchiveEntete entete;
        memcpy(entete.signature, ARCHIVE_SIGNATURE, 4);        
        entete.nombreFichiers = m_elements.size();
    
        fwrite(&entete, sizeof(ArchiveEntete), 1, fichier);
    }
    
    // Ecriture du contenu
    {
        m_iterateur = m_elements.begin();
    
        while (m_iterateur!=m_elements.end())
        {
            // Obtenir l'element courrant
            ArchiveElement element = *m_iterateur;
                        
            // Ecrire l'element
            fwrite(&element, sizeof(ArchiveElement), 1, fichier);
                    
            // Ecrire le contenu de l'element (donnes compresses du fichier)
            switch (element.typeCompression)
            {
                case ID_ARCHIVE_COMPRESSION_AUCUNE:
                {
                    fwrite(element.pointeur, element.tailleOriginale, 1, fichier);
                
                    break;
                };
            
                case ID_ARCHIVE_COMPRESSION_ZLIB:
                {
                    fwrite(element.pointeur, element.tailleCompressee, 1, fichier);
                
                    break;
                };
            }
            
            m_iterateur++;
        }
    }
    
    fclose(fichier);
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- AJOUTER FICHIER
ArchiveResultat Archive::fichierAjouter(string &nomFichier, char *pointeur, int taille)
{
    // Chercher si le fichier existe deja dans l'archive..
    {
        m_iterateur = m_elements.begin();
    
        while (m_iterateur!=m_elements.end())
        {
            ArchiveElement element = *m_iterateur;
            
            if (nomFichier.compare(element.nomFichier)==0)
            {
                // Fichier deja existant !
                return ID_ARCHIVE_ERREUR_FICHIER_DEJA_EXISTANT;
            }
            else
            {
                m_iterateur++;
            }
        }
    }
    
    ArchiveElement element;
    
    // Preparer les informations concernant l'element
    {
        strcpy(element.nomFichier, nomFichier.c_str());
        element.tailleOriginale = taille;
    
        element.pointeur = new char[taille];
        memcpy(element.pointeur, pointeur, taille);
    
        element.typeCompression = ID_ARCHIVE_COMPRESSION_AUCUNE;
    }
    
    // Compresse les donnes (contenu du fichier)
    fichierCompresser(element);
    
    // Rajouter l'element  la liste des elements
    m_elements.push_back(element);
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- SUPPRIMER FICHIER
ArchiveResultat Archive::fichierSupprimer(string &nomFichier)
{
    m_iterateur = m_elements.begin();
    
    while (m_iterateur!=m_elements.end())
    {
        ArchiveElement element = *m_iterateur;
            
        if (nomFichier.compare(element.nomFichier)==0)
        {
            // Element trouv..
            
            delete [] element.pointeur;
            
            m_iterateur = m_elements.erase(m_iterateur);
            
            return ID_ARCHIVE_OK;
        }
        else
        {
            m_iterateur++;
        }
    }
    
    // Le fichier n'a pas t trouv dans l'archive !
    return ID_ARCHIVE_ERREUR_FICHIER_INEXISTANT;
}

// ---------------------------------------------------------------------------- COMPRESSER FICHIER
ArchiveResultat Archive::fichierCompresser(ArchiveElement &element)
{
 	// Cre un fichier temporaire dans lequel nous allons ecrire
    // dedans les donnes compresses.  
    char *nomFichierTemporaire = _tempnam(NULL, NULL);
    
    // Compresse !
    {
        wxFileOutputStream fileOutputStream(nomFichierTemporaire);
        wxZlibOutputStream zlibOutputStream(fileOutputStream);
    
        zlibOutputStream.Write(element.pointeur, element.tailleOriginale);
    }
    
    // Relire le fichier compress et met  jour l'element..
    {
        FILE *fichier = fopen(nomFichierTemporaire, "rb");

        // Obtenir la taille du fichier compress
        fseek(fichier, 0, SEEK_END);
        element.tailleCompressee = ftell(fichier);
        fseek(fichier, 0, SEEK_SET);
    
        // Efface l'ancien pointeur (donnes non compresse)
        delete [] element.pointeur;
                
        // Le nouveau pointeur vers les donnes compresses
        element.pointeur = new char[element.tailleCompressee];
    
        // Rempli le tableau fraichement allou avec les donnes compresses
        fread(element.pointeur, element.tailleCompressee, 1, fichier);
    
        // Ferme le fichier temporaire
        fclose(fichier);
        
        // Efface le fichier temporaire !
        remove(nomFichierTemporaire);
    }
    
    // Specifie le type de compression
    element.typeCompression = ID_ARCHIVE_COMPRESSION_ZLIB;
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- DECOMPRESSER FICHIER
ArchiveResultat Archive::fichierDecompresser(string &nomFichier, char *&pointeur, int &taille)
{
    // Obtenir l'element correspondant au fichier specifi
    ArchiveElement element = fichierObtenirElement(nomFichier);
    
    // Alloue le tableau pour y mettre les donnes decompresses
    // Quelque soit le type de compression utilis, les donnes  decompresser
    // seront toujours les memes !
    taille = element.tailleOriginale;
    pointeur = new char[taille];  
            
    switch (element.typeCompression)
    {
        case ID_ARCHIVE_COMPRESSION_AUCUNE:
        {   
            // Pas de compression, simple recopie..
            memcpy(pointeur, element.pointeur, taille);
            
            break;
        }
        
        case ID_ARCHIVE_COMPRESSION_ZLIB:
        {   
            // Generer un nom de fichier temporaire..
            char *nomFichierTemporaire = _tempnam(NULL, NULL);
            
            // Ecrire dans un fichier temporaire le contenu compress du fichier
            FILE *fichier = fopen(nomFichierTemporaire, "wb");
            fwrite(element.pointeur, element.tailleCompressee, 1, fichier);
            fclose(fichier);
    
            // Decompression en mmoire en utilisant la Zlib
            {
                wxFileInputStream fileInputStream(nomFichierTemporaire);
                wxZlibInputStream zlibInputStream(fileInputStream);
    
                zlibInputStream.Read(pointeur, element.tailleOriginale);
            }
            
            // Efface le fichier temporaire !
            remove(nomFichierTemporaire);
            
            break;
        }
    }
    
    return ID_ARCHIVE_OK;
}

// ---------------------------------------------------------------------------- OBTENIR NOMBRE FICHIERS
int Archive::archiveObtenirNombreFichiers()
{
    // Simple, non ? Vive les STL..
    return m_elements.size();
}

// ---------------------------------------------------------------------------- OBTENIR NOM FICHIER PAR INDICE
string Archive::archiveObtenirNomFichierParIndice(int indice)
{
    // Simple, non ? Vive les STL..
    return m_elements.at(indice).nomFichier;
}
    
// ---------------------------------------------------------------------------- OBTENIR UN ELEMENT DE L'ARCHIVE
ArchiveElement Archive::fichierObtenirElement(string &nomFichier)
{
    m_iterateur = m_elements.begin();
    
    while (m_iterateur!=m_elements.end())
    {
        ArchiveElement element = *m_iterateur;
            
        if (nomFichier.compare(element.nomFichier)==0)
        {
            // L'element a t trouv, on le retourne..
            return element;
        }
        else
        {
            m_iterateur++;
        }
    }
    
    // Element non trouv !
    ArchiveElement element;
    memset(&element, 0, sizeof(ArchiveElement));
    
    return element;
}
